<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Pagination\LengthAwarePaginator;
use App\Models\Product;
use App\Models\ProductBillingual;
use Image;

class ProductBillingualController extends Controller
{
    //get all data
    public function getAllData() {
        $dataArray = array();
        $dataProducts = Product::orderByDesc('id')->get();
        $countDataProduct = Product::count();

        foreach($dataProducts as $dataProduct) {
            $dataProductBillingual = ProductBillingual::where('id_product', $dataProduct->id)->first();
            array_push($dataArray, (object) [
                'product' => $dataProduct,
                'productBillingual' => $dataProductBillingual
            ]);
        }

        return response()->json([
            'status' => true,
            'message' => 'Data Produk berhasil didapatkan',
            'totalData' => $countDataProduct,
            'data' => $dataArray
        ], 200);
    }

    //get all data with pagination
    public function getAllDataWithPagination(Request $request) {
        $dataArray = array();
        $queryDataProducts = Product::orderByDesc('id');
        $dataProducts = $queryDataProducts->get();
        $totalDataProduct = $queryDataProducts->count();

        foreach($dataProducts as $dataProduct) {
            $dataProductBillingual = ProductBillingual::where('id_product', $dataProduct->id)->first();
            array_push($dataArray, (object)[
                'product' => $dataProduct,
                'productBillingual' => $dataProductBillingual
            ]);
        }

        $perPage = 20;
        $currentPage = $request->currentPage;
        $offset = ($currentPage * $perPage) - $perPage;
        $dataToShow = array_slice($dataArray, $offset, $perPage);
        
        $dataArrayPaginate = new LengthAwarePaginator($dataToShow, $totalDataProduct, $perPage);

        return response()->json([
            'status' => true,
            'data' => $dataArrayPaginate
        ], 200);
    }

    //select data by first and highlight
    public function selectDataByFirstAndHighlight() {
        $mainDataArray = array();
        $firstDataArray = array();
        $highlightDataArray = array();
        $dataProductFirst = Product::where('highlight', 1)->first();
        $dataProductFirstBillingual = ProductBillingual::where('id_product', $dataProductFirst->id)->first();
        $dataProductHighlights = Product::orderByDesc('id')->where('highlight', 2)->get();

        array_push($firstDataArray, (object) [
            'product' => $dataProductFirst,
            'productBillingual' => $dataProductFirstBillingual
        ]);

        foreach($dataProductHighlights as $dataProductHighlight) {
            $dataProductHighlightBillingual = ProductBillingual::where('id_product', $dataProductHighlight->id)->first();
            array_push($highlightDataArray, (object) [
                'product' => $dataProductHighlight,
                'productBillingual' => $dataProductHighlightBillingual
            ]);
        }

        array_push($mainDataArray, (object) [
            'first' => $firstDataArray,
            'highlight' => $highlightDataArray
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Data Produk berhasil didapatkan',
            'data' => $mainDataArray
        ], 200);
    }

    //select data by id
    public function selectDataById(Request $request) {
        $dataArray = array();
        $dataProduct = Product::where('id', $request->id)->first();
        $dataProductBillingual = ProductBillingual::where('id_product', $request->id)->first();

        array_push($dataArray, (object) [
            'product' => $dataProduct,
            'productBillingual' => $dataProductBillingual
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Data Produk berhasil didapatkan',
            'data' => $dataArray
        ], 200);
    }

    //select data by slug
    public function selectDataBySlug(Request $request) {
        $dataArray = array();
        $dataProduct = Product::where('slug', $request->slug)->first();
        $dataProductBillingual = ProductBillingual::where('id_product', $dataProduct->id)->first();

        array_push($dataArray, (object) [
            'product' => $dataProduct,
            'productBillingual' => $dataProductBillingual
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Data Produk berhasil didapatkan',
            'data' => $dataArray
        ], 200);
    }

    //store data
    public function storeData(Request $request) {
        if($request->hasFile('image')){
            $validateImage = $this->validate($request, [
                'image' => 'required|image|mimes:jpg,png,jpeg|max:1048',
            ]);

            $image = $request->file('image');
            $imageName = time().'.'.$image->extension();
            $thumbPath = public_path('/storage/product/thumb_images');
            $destinationPath = public_path('/storage/product/images');
            $img = Image::make($image->path());
            $thumbImage = $img->resize(600, 600, function($constraint) {
                $constraint->aspectRatio();
            })->save($thumbPath.'/thumb_'.$imageName);
            $mainImage = $image->move($destinationPath, $imageName);
        }else{
            $imageName = 'no-image';
        }

        $countDataProduct = Product::count();

        if($countDataProduct == 0) {
            $highlight = 1;
        } else if($countDataProduct < 3) {
            $highlight = 2;
        } else {
            $highlight = 0;
        }

        $storeProduct = Product::create([
            'name' => $request->name,
            'slug' => $request->slug,
            'text' => $request->text,
            'highlight' => $highlight,
            'thumb_image' => 'thumb_'.$imageName,
            'image' => $imageName
        ]);

        ProductBillingual::create([
            'name' => $request->nameBilingual,
            'text' => $request->textBilingual,
            'id_product' => $storeProduct->id
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Data produk behasil disimpan'
        ], 200);
    }

    //update data
    public function updateData(Request $request) {
        $data = Product::where('id', $request->id)->first();

        if($request->hasFile('image')){
            $validateImage = $this->validate($request, [
                'image' => 'required|image|mimes:jpg,png,jpeg|max:1048',
            ]);

            $image = $request->file('image');
            $imageName = time().'.'.$image->extension();
            $thumbPath = public_path('/storage/product/thumb_images');
            $destinationPath = public_path('/storage/product/images');
            $img = Image::make($image->path());

            $thumbImage = $img->resize(600, 600, function($constraint) {
                $constraint->aspectRatio();
            })->save($thumbPath.'/thumb_'.$imageName);

            $mainImage = $image->move($destinationPath, $imageName);

            File::delete($thumbPath.'/'.$data->thumb_image);
            File::delete($destinationPath.'/'.$data->image);
        }else{
            $imageName = $data->image;
        }

        Product::where('id', $request->id)->update([
            'name' => $request->name,
            'slug' => $request->slug,
            'text' => $request->text,
            'thumb_image' => 'thumb_'.$imageName,
            'image' => $imageName
        ]);

        ProductBillingual::where('id_product', $request->id)->update([
            'name' => $request->nameBilingual,
            'text' => $request->textBilingual,
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Data produk behasil diupdate'
        ], 200);
    }

    //update hightlight
    public function updateDataHighlight(Request $request) {
        $dataProductHighlights = Product::where('highlight', 2)->get();

        foreach($dataProductHighlights as $dataProductHighlight) {
            Product::where('highlight', $dataProductHighlight->highlight)->update([
                'highlight' => 0
            ]);
        }

        foreach(json_decode($request->id) as $id) {
            Product::where('id', $id)->update([
                'highlight' => 2
            ]);
        }

        return response()->json([
            'status' => true,
            'message' => 'Data produk behasil dijadikan highlight'
        ], 200); 
    }

    //update first
    public function updateDataFirst(Request $request) {
        $query = Product::where('id', $request->id[0]);
        $dataProduct = $query->first();

        if($dataProduct->highlight == 2) {
            Product::where('highlight', 1)->update([
                'highlight' => 2
            ]);
            $query->update([
                'highlight' => 1
            ]);
        } else {
            Product::where('highlight', 1)->update([
                'highlight' => 0
            ]);
            Product::where('id', $request->id[0])->update([
                'highlight' => 1
            ]);
        }

        return response()->json([
            'status' => true,
            'message' => 'Data produk behasil dijadikan urutan pertama',
        ], 200);
    }

    //delete data
    public function deleteData(Request $request) {
        $dataProduct = Product::where('id', $request->id)->first();

        if($dataProduct->highlight == 1 || $dataProduct->highlight == 2) {
            return response()->json([
                'status' => false,
                'message' => 'Data Produk tidak bisa dihapus karena dalam keadaan dihighlight'
            ], 200);
        }

        $pathThumbImage = public_path('/storage/product/thumb_images');
        $pathImage = public_path('/storage/product/images');
        File::delete($pathThumbImage.'/'.$dataProduct->thumb_image);
        File::delete($pathImage.'/'.$dataProduct->image);

        Product::where('id', $request->id)->delete();
        ProductBillingual::where('id_product', $request->id)->delete();

        return response()->json([
            'status' => true,
            'message' => 'Data product berhasil dihapus'
        ], 200);
    }
}
